<?php
/**
 * Strata Three functions and definitions.
 *
 * @link https://developer.wordpress.org/themes/basics/theme-functions/
 *
 * @package Strata_Three
 */


if ( ! function_exists( 's3_setup' ) ) :
/**
 * Sets up theme defaults and registers support for various WordPress features.
 *
 * Note that this function is hooked into the after_setup_theme hook, which
 * runs before the init hook. The init hook is too late for some features, such
 * as indicating support for post thumbnails.
 */
function s3_setup() {
	/*
	 * Make theme available for translation.
	 * Translations can be filed in the /languages/ directory.
	 * If you're building a theme based on Strata Three, use a find and replace
	 * to change 's3' to the name of your theme in all the template files.
	 */
	load_theme_textdomain( 's3', get_template_directory() . '/languages' );

	// Add default posts and comments RSS feed links to head.
	add_theme_support( 'automatic-feed-links' );

	/*
	 * Let WordPress manage the document title.
	 * By adding theme support, we declare that this theme does not use a
	 * hard-coded <title> tag in the document head, and expect WordPress to
	 * provide it for us.
	 */
	add_theme_support( 'title-tag' );

	/*
	 * Enable support for Post Thumbnails on posts and pages.
	 *
	 * @link https://developer.wordpress.org/themes/functionality/featured-images-post-thumbnails/
	 */
	add_theme_support( 'post-thumbnails' );


	// This theme uses wp_nav_menu() in one location.
	register_nav_menus( array(
		'primary' => esc_html__( 'Primary Menu', 's3' ),
	) );

	/*
	 * Switch default core markup for search form, comment form, and comments
	 * to output valid HTML5.
	 */
	add_theme_support( 'html5', array(
		'search-form',
		'comment-form',
		'comment-list',
		'gallery',
		'caption',
	) );

}
endif; // s3_setup
add_action( 'after_setup_theme', 's3_setup' );

/**
 * Set the content width in pixels, based on the theme's design and stylesheet.
 *
 * Priority 0 to make it available to lower priority callbacks.
 *
 * @global int $content_width
 */
function s3_content_width() {
	$GLOBALS['content_width'] = apply_filters( 's3_content_width', 945 );
}
add_action( 'after_setup_theme', 's3_content_width', 0 );

/**
 * Register widget area.
 *
 * @link https://developer.wordpress.org/themes/functionality/sidebars/#registering-a-sidebar
 */
function s3_widgets_init() {
	# Create four (8) sidebars
	$sidebar_count = 8;
	$args = array (
		'name'          => esc_html__( 'Sidebar: %d', 's3' ),
		'id'            => 'sidebar',
		'description'   => '',
		'before_widget' => '<aside id="%1$s" class="widget %2$s">',
		'after_widget'  => '</aside>',
		'before_title'  => '<h2 class="h5">',
		'after_title'   => '</h2>',
	);
	register_sidebars($sidebar_count, $args);
	# Note: Sidebars are name as such: sidebar, sidebar-2, sidebar-3, sidebar-4, etc.
	# (Personal note to the above: this vexes me.)
}
add_action( 'widgets_init', 's3_widgets_init' );

/**
 * Enqueue scripts and styles.
 */
function s3_scripts() {
	$wp_theme   = wp_get_theme( 's3' );
	$wp_version = $wp_theme->get( 'Version' );
	
	# Styles
	wp_enqueue_style( 'style-bootstrap',          get_template_directory_uri() . '/s3.3/css/bootstrap.min.css' );
	wp_enqueue_style( 'style-strata',             get_template_directory_uri() . '/s3.3/css/strata.css', 'style-bootstrap', $wp_version);
	wp_enqueue_style( 'style-cookie',             get_template_directory_uri() . '/s3.3/css/cookie.css', 'style-cookie', $wp_version);
	wp_enqueue_style( 'style-strata-extend',      get_template_directory_uri() . '/s3.3/css/strata-extend.min.css', 'style-extend', $wp_version);
	wp_enqueue_style( 'style-175',                get_template_directory_uri() . '/s3.3/css/175.css', 'style-275', $wp_version);
	wp_enqueue_style( 'style-theme',              get_stylesheet_uri() );
	wp_enqueue_style( 'style-print',              get_template_directory_uri() . '/s3.3/css/print.css', 'style-theme', $wp_version, 'print' );

	# Scripts
  wp_enqueue_script( 'script-jquery',           'https://code.jquery.com/jquery-2.2.4.min.js', array(), $wp_version, true );
	wp_enqueue_script( 'script-jquery-ui',        'https://code.jquery.com/ui/1.12.1/jquery-ui.min.js', array(), $wp_version, true );
	wp_enqueue_script( 'script-bootstrap',        get_template_directory_uri() . '/s3.3/js/bootstrap.min.js', array(), $wp_version, true );
	wp_enqueue_script( 'script-strata',           get_template_directory_uri() . '/s3.3/js/strata.js', array(), $wp_version, true );
	wp_enqueue_script( 'script-osano',           'https://cmp.osano.com/AzZd6cTcYkGFq2Mk6/b22bc8e4-1afb-45b2-8ada-b4941d932758/osano.js', array(), $wp_version, true );
	wp_enqueue_script( 'script-cookie',           get_template_directory_uri() . '/s3.3/js/cookie.js', array(), $wp_version, true );
    wp_enqueue_script( 'script-strata-extend',    get_template_directory_uri() . '/s3.3/js/strata-extend.min.js', array(), $wp_version, true );
	wp_enqueue_script( 'script-navigation',       get_template_directory_uri() . '/js/functions.js', array( 'jquery' ), '20120206', true );

	# Gallery: Frecso
	if (_SITE_GALLERY_TYPE == 'fresco') {
		wp_enqueue_script( 'script-fresco',          get_template_directory_uri() . '/inc/lib/fresco/fresco.js', array(), $wp_version, true );
		wp_enqueue_style(  'style-fresco',           get_template_directory_uri() . '/inc/lib/fresco/fresco.css', 'style-theme', $wp_version);
	}

	wp_enqueue_script( 'fo-skip-link-focus-fix', get_template_directory_uri() . '/js/skip-link-focus-fix.js', array(), $wp_version, true );

	if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
		wp_enqueue_script( 'comment-reply' );
	}

}
add_action( 'wp_enqueue_scripts', 's3_scripts' );

# Custom template tags for this theme.
require get_template_directory() . '/inc/template-tags.php';

# Customizer additions.
#require get_template_directory() . '/inc/customizer.php';

# Load Jetpack compatibility file.
require get_template_directory() . '/inc/jetpack.php';


# Remove iframe from RSS feed
function rss_noiframe($content) {
    $content = preg_replace( '/<iframe(.*)\/iframe>/is', '', $content );

    return $content;
}
add_filter('the_excerpt_rss', 'rss_noiframe');
add_filter('the_content_feed', 'rss_noiframe');



# Remove iframe's frameborder property
function wpse_308247_remove_frameborder( $html, $url ) {
    // If the URL to be embedded is from YouTube.
    if ( strpos( $url, 'youtube.com' ) !== false ) {
        // Replace the frameborder attribute with an empty string.
        $html = str_replace( 'frameborder="0"', '', $html );
    }

    return $html;
}
add_filter( 'embed_oembed_html', 'wpse_308247_remove_frameborder', 10, 2 );



# Remove image stretching caused by WordPress 6.7.1
remove_action( 'wp_head', 'wp_print_auto_sizes_contain_css_fix', 1 );
add_filter(
  'wp_content_img_tag',
  static function ( $image ) {
    return str_replace( ' sizes="auto, ', ' sizes="', $image );
  }
);
add_filter(
  'wp_get_attachment_image_attributes',
  static function ( $attr ) {
    if ( isset( $attr['sizes'] ) ) {
      $attr['sizes'] = preg_replace( '/^auto, /', '', $attr['sizes'] );
    }
    return $attr;
  }
);



/**
 * YouTube, Vimeo, and Soundcloud embed helpers
 */

// --------------------------------------------------------------------------------------------------
// YOUTUBE IFRAME GENERATOR
// --------------------------------------------------------------------------------------------------
function generate_youtube_iframe($video_id) {
    // Includes: class="youtube-video", ?enablejsapi=1, role="region", aria-label
    $default_title = 'Embedded YouTube video'; 
    
    // FIX 1: Removed deprecated showinfo=0.
    // FIX 2: Moved title attribute into the main tag (it was outside the tag structure).
    // FIX 3: Ensures title and aria-label are set initially for accessibility checkers.
    return '<div class="video">
        <iframe 
            class="youtube-video" 
            width="640" 
            height="360" 
            src="https://www.youtube.com/embed/' . esc_attr($video_id) . '?rel=0&amp;enablejsapi=1" 
            frameborder="0" 
            allowfullscreen="allowfullscreen" 
            role="region" 
            aria-label="' . esc_attr($default_title) . '" 
            title="' . esc_attr($default_title) . '">
        </iframe>
    </div>';
}

// --------------------------------------------------------------------------------------------------
// SOUNDCLOUD IFRAME GENERATOR
// --------------------------------------------------------------------------------------------------
function generate_soundcloud_iframe($track_id) {
    // Includes: class="soundcloud-embed"
    $default_title = 'Embedded Soundcloud audio'; 
    
    // FIX 4: Added role="region" and aria-label for accessibility compliance.
    return '<div class="audio">
        <iframe 
            class="soundcloud-embed"
            width="100%" 
            height="166" 
            scrolling="no" 
            frameborder="no" 
            src="https://w.soundcloud.com/player/?url=https%3A//api.soundcloud.com/tracks/' . esc_attr($track_id) . '&amp;color=ceb888&amp;auto_play=false&amp;hide_related=true&amp;show_comments=false&amp;show_user=true&amp;show_reposts=false"
            role="region" 
            aria-label="' . esc_attr($default_title) . '"
            title="' . esc_attr($default_title) . '">
        </iframe>
    </div>';
}

// --------------------------------------------------------------------------------------------------
// VIMEO IFRAME GENERATOR
// --------------------------------------------------------------------------------------------------
function generate_vimeo_iframe($video_id) {
    // Includes: class="vimeo-embed" and ?api=1&player_id=...
    $default_title = 'Embedded Vimeo video'; 
    $player_id = 'vimeo-player-' . esc_attr($video_id);
    
    // FIX 5: Added role="region" and aria-label for accessibility compliance.
    return '<div class="video">
        <iframe 
            class="vimeo-embed" 
            id="' . $player_id . '"
            src="https://player.vimeo.com/video/' . esc_attr($video_id) . '?title=0&byline=0&portrait=0&api=1&player_id=' . $player_id . '" 
            width="640" 
            height="360" 
            frameborder="0" 
            allowfullscreen="allowfullscreen"
            role="region"
            aria-label="' . esc_attr($default_title) . '"
            title="' . esc_attr($default_title) . '">
        </iframe>
    </div>';
}


/**
 * Scans post content for raw URLs and [embed] shortcodes (Part 1).
 * Finds and fixes attributes on ALL existing media iframes (Part 2), 
 * ensuring required classes, role, and aria-label are present regardless of source.
 */
function fix_user_media_embeds($content) {
    
    // --- PART 1: Convert raw URLs/Shortcodes to clean iframes (Relies on generator functions) ---
    
    // 1. YOUTUBE, SOUNDCLOUD, VIMEO Regex patterns must be defined here if they were outside.
    // Assuming they are defined and the $process_link function is defined like this:
    
    $process_link = function($matches) {
        $link = $matches[0];

        // 1. CHECK FOR YOUTUBE
        if (preg_match('#(?:youtu\.be/|youtube\.com/(?:watch\?v=|v/|embed/))([a-zA-Z0-9_-]{11})#i', $link, $id_matches)) {
            return generate_youtube_iframe($id_matches[1]);
        }
        // 2. CHECK FOR VIMEO
        if (preg_match('#vimeo\.com/([0-9]+)#i', $link, $id_matches)) {
            return generate_vimeo_iframe($id_matches[1]);
        }
        // 3. CHECK FOR SOUNDCLOUD
        if (preg_match('#tracks/([0-9]+)#i', $link, $id_matches)) {
            return generate_soundcloud_iframe($id_matches[1]);
        }
        return $link;
    };

    // Apply the link/shortcode conversion
    $content = preg_replace_callback('/\[embed\]\s*([^\]\s]+)\s*\[\/embed\]/i', $process_link, $content);
    $content = preg_replace_callback('#\s(https?://(?:www\.)?(?:youtube\.com|youtu\.be|vimeo\.com|soundcloud\.com)[^\s\<\>]+)\s#i', $process_link, $content);

    // --- PART 2: Find and forcibly fix ALL existing iframes (oEmbed, ACF, etc.) ---
    
    // Find all iframe tags in the content
    $content = preg_replace_callback('/<iframe[^>]*>/i', function($matches) {
        $full_tag = $matches[0];
        
        $class_to_add = '';
        $default_title = "Embedded Media Content";

        // Determine media type based on source URL
        if (strpos($full_tag, 'youtube.com') !== false || strpos($full_tag, 'youtu.be') !== false) {
            $class_to_add = 'youtube-video';
            $default_title = "Embedded YouTube Video";
        } elseif (strpos($full_tag, 'vimeo.com') !== false) {
            $class_to_add = 'vimeo-embed';
            $default_title = "Embedded Vimeo Video";
        } elseif (strpos($full_tag, 'soundcloud.com') !== false) {
            $class_to_add = 'soundcloud-embed';
            $default_title = "Embedded Soundcloud Audio";
        }
        
        // If a supported media source was found, ensure it has the required attributes:
        if ($class_to_add) {
            $new_attributes = '';
            
            // 1. Inject Class: Check if the class is already present.
            if (strpos($full_tag, 'class="' . $class_to_add . '"') === false) {
                // If missing, add the class. This is the fix for oEmbeds.
                $new_attributes .= ' class="' . $class_to_add . '"';
            }
            
            // 2. Inject Role: Check if role="region" is present.
            if (strpos($full_tag, 'role="region"') === false) {
                $new_attributes .= ' role="region"';
            }
            
            // 3. Inject aria-label (Fallback): Check if aria-label is present.
            if (strpos($full_tag, 'aria-label=') === false) {
                // Use esc_attr to safely escape the title text.
                $new_attributes .= ' aria-label="' . esc_attr($default_title) . '"';
            }

            // If we have new attributes to add, inject them into the start of the tag.
            if (!empty($new_attributes)) {
                // Replace the opening '<iframe' with '<iframe' + all the new attributes
                return str_replace('<iframe', '<iframe' . $new_attributes, $full_tag);
            }
        }
        
        return $full_tag; // Return the original tag if no changes were needed
    }, $content);
    
    return $content;
}
// Use a late priority (15) to run after oEmbed conversion is complete.
add_filter('the_content', 'fix_user_media_embeds', 15);